<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Customer;
use App\Models\Restaurant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class RestaurantController extends Controller
{

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if(Auth::user()->role=='admin'){
            $restaurants = Restaurant::with('user')->latest()->get();
            return view('restaurant.index', compact('restaurants'));
        } else {
            $customers = Customer::latest()->paginate(15);

            $locations = Customer::select('location')
                ->selectRaw('COUNT(*) as total')
                ->groupBy('location')
                ->get();

            $totalCustomer= Customer::count();

            return view('customer.customer-view', compact('customers', 'locations', 'totalCustomer'));
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('restaurant.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator= Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'image' => 'required|image|mimes:jpeg,png,jpg|max:5120',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|confirmed|min:6',
            'address' => 'nullable|string',
            'website' => 'nullable|url',
        ]);

        if($validator->fails()){
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Create User
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        // Handle Image Upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $destinationPath = public_path('assets/uploads/restaurant_img');
            $image->move($destinationPath, $imageName);
            $imagePath = 'assets/uploads/restaurant_img/'. $imageName;
        } else {
            $imagePath = null;
        }

        // Create Restaurant associated with User
        $restaurant = Restaurant::create([
            'user_id' => $user->id,
            'address' => $request->address,
            'website' => $request->website,
            'image' => $imagePath,
        ]);

        return redirect('/restaurants')->with('success', 'Restaurant created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $restaurant = Restaurant::with('user')->findOrFail($id);
        return view('restaurant.edit', compact('restaurant'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $restaurant = Restaurant::with('user')->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $restaurant->user->id,
            'password' => 'nullable|confirmed|min:6',
            'image' => 'nullable|image|mimes:jpeg,png,jpg|max:5120', // optional new image
            'address' => 'nullable|string',
            'website' => 'nullable',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Update User
        $restaurant->user->update([
            'name' => $request->name,
            'email' => $request->email,
            'password' => $request->password ? Hash::make($request->password) : $restaurant->user->password,
        ]);

        // Handle Image Upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($restaurant->image && file_exists(public_path($restaurant->image))) {
                unlink(public_path($restaurant->image));
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $destinationPath = public_path('assets/uploads/restaurant_img'); // folder bhi yahan ho
            $image->move($destinationPath, $imageName);
            $imagePath = 'assets/uploads/restaurant_img/' . $imageName; // same path DB me

            $restaurant->image = $imagePath;
        }

        // Update Restaurant
        $restaurant->address = $request->address;
        $restaurant->website = $request->website;
        $restaurant->save();

        return redirect()->route('restaurants.index')->with('success', 'Restaurant updated successfully!');
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
